<?php
session_start();
if (!isset($_SESSION['admin_logged_in'])) {
    header("Location: admin_login.php");
    exit();
}

// Include database connection
require 'db_connect.php';

// Fetch submission trends (last 12 months)
$submissionTrendsQuery = $pdo->query("
    SELECT DATE_FORMAT(submitted_at, '%Y-%m') AS month, COUNT(*) AS count 
    FROM car_submissions 
    GROUP BY month 
    ORDER BY month DESC 
    LIMIT 12
");
$submissionTrends = $submissionTrendsQuery->fetchAll(PDO::FETCH_ASSOC);

// Fetch submission status counts
$statusCountsQuery = $pdo->query("
    SELECT status, COUNT(*) AS count 
    FROM car_submissions 
    GROUP BY status
");
$statusCounts = $statusCountsQuery->fetchAll(PDO::FETCH_ASSOC);

// Fetch Equipe Series distribution
$equipeSeriesQuery = $pdo->query("
    SELECT equipe_series, COUNT(*) AS count 
    FROM car_submissions 
    GROUP BY equipe_series 
    ORDER BY count DESC
");
$equipeSeriesData = $equipeSeriesQuery->fetchAll(PDO::FETCH_ASSOC);

$equipeSeriesLabels = json_encode(array_column($equipeSeriesData, 'equipe_series'));
$equipeSeriesCounts = json_encode(array_column($equipeSeriesData, 'count'));


// Fetch top vehicle models submitted
$topVehiclesQuery = $pdo->query("
    SELECT vehicle_model, COUNT(*) AS count 
    FROM car_submissions 
    GROUP BY vehicle_model 
    ORDER BY count DESC 
    LIMIT 5
");
$topVehicles = $topVehiclesQuery->fetchAll(PDO::FETCH_ASSOC);

// ✅ Format data properly (Prevents trailing commas)
$submissionTrendsLabels = json_encode(array_column($submissionTrends, 'month'));
$submissionTrendsCounts = json_encode(array_column($submissionTrends, 'count'));


$statusLabels = json_encode(array_column($statusCounts, 'status'));
$statusCountsData = json_encode(array_column($statusCounts, 'count'));

$topVehicleLabels = json_encode(array_column($topVehicles, 'vehicle_model'));
$topVehicleCounts = json_encode(array_column($topVehicles, 'count'));

// Fetch submission counts
$totalSubmissions = $pdo->query("SELECT COUNT(*) FROM car_submissions")->fetchColumn();
$approvedSubmissions = $pdo->query("SELECT COUNT(*) FROM car_submissions WHERE status = 'Approved'")->fetchColumn();
$pendingSubmissions = $pdo->query("SELECT COUNT(*) FROM car_submissions WHERE status = 'Pending'")->fetchColumn();
$rejectedSubmissions = $pdo->query("SELECT COUNT(*) FROM car_submissions WHERE status = 'Rejected'")->fetchColumn();

// Fetch submissions
$query = $pdo->query("SELECT * FROM car_submissions ORDER BY submitted_at DESC");
$submissions = $query->fetchAll(PDO::FETCH_ASSOC);
// Fetch unread notifications count
$notificationCountStmt = $pdo->query("SELECT COUNT(*) FROM admin_notifications WHERE is_read = 0");
$unreadNotifications = $notificationCountStmt->fetchColumn();

// Set number of records per page
$limit = 10;
$page = isset($_GET['page']) && is_numeric($_GET['page']) ? intval($_GET['page']) : 1;
$offset = ($page - 1) * $limit;

// Get total number of submissions for pagination
$totalSubmissionsQuery = $pdo->query("SELECT COUNT(*) FROM car_submissions");
$totalSubmissions = $totalSubmissionsQuery->fetchColumn();
$totalPages = ceil($totalSubmissions / $limit);

// Fetch paginated submissions
$recentSubmissionsQuery = $pdo->prepare("
    SELECT id, owner_name, submitted_at 
    FROM car_submissions 
    ORDER BY submitted_at DESC 
    LIMIT :limit OFFSET :offset
");

$recentSubmissionsQuery->bindParam(':limit', $limit, PDO::PARAM_INT);
$recentSubmissionsQuery->bindParam(':offset', $offset, PDO::PARAM_INT);
$recentSubmissionsQuery->execute();
$recentSubmissions = $recentSubmissionsQuery->fetchAll(PDO::FETCH_ASSOC);

?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard</title>
    <link rel="stylesheet" href="admin_styles.css">
     <!-- ✅ Ensure Chart.js loads first -->
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>

</head>
<body>

   <?php include 'sidebar.php'; ?> <!-- Sidebar is included here -->

    <!-- Main Content -->
<main class="admin-container">
    <h1>Admin Dashboard</h1>
    <h2>Status Breakdown</h2>

    <div class="stats-container">
        <div class="stat-box total">
            <h3>Total Submissions</h3>
            <p><?= $totalSubmissions ?></p>
        </div>
        <div class="stat-box approved">
            <h3>Approved</h3>
            <p><?= $approvedSubmissions ?></p>
        </div>
        <div class="stat-box pending">
            <h3>Pending</h3>
            <p><?= $pendingSubmissions ?></p>
        </div>
        <div class="stat-box rejected">
            <h3>Rejected</h3>
            <p><?= $rejectedSubmissions ?></p>
        </div>
    </div> <!-- ✅ Correct Closing -->

    <!-- ✅ Recent Activity - Timeline Style -->
    <div class="activity-feed">
        <h2>Recent Activity</h2>
        <ul class="activity-timeline">
            <?php if (!empty($recentSubmissions)): ?>
                <?php foreach ($recentSubmissions as $submission): ?>
                    <li>
                        <div class="timeline-dot"></div>
                        <div class="timeline-content">
                            <strong>
                                <a href="view_submission.php?id=<?= $submission['id'] ?>" class="submission-link">
                                    <?= htmlspecialchars($submission['owner_name']) ?>
                                </a>
                            </strong> 
                            submitted a car <br>
                            <small><?= date("M d, Y H:i", strtotime($submission['submitted_at'])) ?></small>
                        </div>
                    </li>
                <?php endforeach; ?>
            <?php else: ?>
                <p>No recent activity.</p>
            <?php endif; ?>
        </ul>

        <!-- ✅ Pagination Links Inside Correct Div -->
        <div class="pagination">
            <?php if ($page > 1): ?>
                <a href="?page=<?= $page - 1 ?>">« Prev</a>
            <?php endif; ?>

            <?php for ($i = 1; $i <= $totalPages; $i++): ?>
                <a href="?page=<?= $i ?>" class="<?= ($i == $page) ? 'active' : '' ?>"><?= $i ?></a>
            <?php endfor; ?>

            <?php if ($page < $totalPages): ?>
                <a href="?page=<?= $page + 1 ?>">Next »</a>
            <?php endif; ?>
        </div>
    </div> 

    <div class="submissions-container">
        <h2>Latest Submissions</h2>

        <!-- Search, Filter & Export -->
        <div class="filter-container">
            <input type="text" id="search" placeholder="Search submissions..." onkeyup="searchTable()">
            <label for="filterStatus">Filter by Status:</label>
            <select id="filterStatus" onchange="filterTable()">
                <option value="">All Statuses</option>
                <option value="Pending">Pending</option>
                <option value="Approved">Approved</option>
                <option value="Rejected">Rejected</option>
            </select>
            <button onclick="exportCSV()">Export CSV</button>
            <button onclick="exportPDF()">Export PDF</button>
        </div>

        <!-- ✅ Submission Table -->
        <table class="admin-table" id="submissionsTable">
            <thead>
                <tr>
                    <th>ID</th>
                    <th>Owner Name</th>
                    <th>Vehicle</th>
                    <th>Equipe Series</th>
                    <th>Equipe Class</th>
                    <th>Status</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
    <?php if (!empty($submissions)): ?>
        <?php foreach ($submissions as $submission): ?>
            <tr>
                <td><?= htmlspecialchars($submission['id']) ?></td>
                <td><?= htmlspecialchars($submission['owner_name']) ?></td>
                <td><?= htmlspecialchars($submission['vehicle_model']) ?></td>
                <td><?= htmlspecialchars($submission['equipe_series']) ?></td>
                <td><?= htmlspecialchars($submission['equipe_class']) ?></td>
                <td>
                    <select class="status-dropdown" data-id="<?= $submission['id'] ?>">
                        <option value="Pending" <?= $submission['status'] == 'Pending' ? 'selected' : '' ?>>Pending</option>
                        <option value="Approved" <?= $submission['status'] == 'Approved' ? 'selected' : '' ?>>Approved</option>
                        <option value="Rejected" <?= $submission['status'] == 'Rejected' ? 'selected' : '' ?>>Rejected</option>
                    </select>
                </td>
                <td>
                    <a href="view_submission.php?id=<?= $submission['id'] ?>" class="view-btn">View</a>
                    <a href="edit_submission.php?id=<?= $submission['id'] ?>" class="edit-btn">Edit</a>
                    <a href="export_pdf.php?id=<?= $submission['id'] ?>" class="export-btn">Export PDF</a>
                </td>
            </tr>
        <?php endforeach; ?>
    <?php else: ?>
        <tr>
            <td colspan="5">No submissions found.</td>
        </tr>
    <?php endif; ?>
</tbody>
        </table>

        <!-- ✅ "View All" Button Properly Inside -->
        <div class="view-all-container">
            <a href="submissions.php" class="view-all-btn">View All Submissions</a>
        </div>
    </div> <!-- ✅ Properly Closed -->



        <div class="analytics-container">
    <h2>📊 Analytics Overview</h2>
    <div class="analytics-grid">
        <div class="date-filter" style="margin-bottom: 20px; text-align: center;">
    <label for="startDate">From:</label>
    <input type="date" id="startDate">
    <label for="endDate">To:</label>
    <input type="date" id="endDate">
    <button onclick="applyDateFilter()">Apply</button>
</div>

        <!-- Submission Trends -->
        <div class="analytics-card">
            <h3>📅 Submissions Per Month</h3>
            <canvas id="submissionTrendsChart"></canvas>
        </div>

        <!-- Submission Status Pie Chart -->
        <div class="analytics-card">
            <h3>📌 Submission Status Breakdown</h3>
            <canvas id="statusPieChart"></canvas>
        </div>

        <!-- Top Vehicle Models -->
        <div class="analytics-card">
            <h3>🚗 Top Submitted Vehicle Models</h3>
            <canvas id="topVehiclesChart"></canvas>
        </div>

        <!-- Most Common Submission Time -->
        <div class="analytics-card">
            <h3>⏳ Peak Submission Times</h3>
            <canvas id="submissionTimeChart"></canvas>
        </div>

        <!-- Submission Rate -->
        <div class="analytics-card">
            <h3>📈 Average Submissions Per Day</h3>
            <p id="avgSubmissions"></p>
        </div>

        <!-- Approval vs Rejection Rate -->
        <div class="analytics-card">
            <h3>✅ Approval Rate vs ❌ Rejection Rate</h3>
            <canvas id="approvalRateChart"></canvas>
        </div>
        <!-- Equipe Series Bar Chart -->
        <div class="analytics-card">
            <h3>🏁 Submissions by Equipe Series</h3>
             <canvas id="equipeSeriesChart"></canvas>
        </div>
        </div>
</div>


    </main>

<script>
    // ✅ SEARCH FUNCTION (Fixes issue where it wasn't checking columns properly)
    function searchTable() {
        let input = document.getElementById("search").value.toUpperCase();
        let table = document.getElementById("submissionsTable");
        let rows = table.getElementsByTagName("tr");

        for (let i = 1; i < rows.length; i++) {
            let showRow = false;
            let cells = rows[i].getElementsByTagName("td");

            for (let j = 0; j < cells.length; j++) {
                let cellText = cells[j].textContent || cells[j].innerText;
                if (cellText.toUpperCase().includes(input)) {
                    showRow = true;
                    break;
                }
            }
            rows[i].style.display = showRow ? "" : "none";
        }
    }

   

    $(document).ready(function () {
    // ✅ FIX: Status Change Event (Update Database on Change)
    $(".status-dropdown").change(function () {
        let submissionId = $(this).data("id");
        let newStatus = $(this).val();

        $.post("update_status.php", { id: submissionId, status: newStatus }, function (response) {
            alert("✅ Status updated successfully!");
        }).fail(function () {
            alert("❌ Error updating status!");
        });
    });

    // ✅ FIX: Filter Submissions by Status
    function filterTable() {
        let selectedStatus = $("#filterStatus").val().toLowerCase();
        $("#submissionsTable tbody tr").each(function () {
            let rowStatus = $(this).find(".status-dropdown").val().toLowerCase();
            if (selectedStatus === "" || rowStatus === selectedStatus) {
                $(this).show();
            } else {
                $(this).hide();
            }
        });
    }

    // ✅ Bind Filter to Change Event
    $("#filterStatus").change(filterTable);
});

    // ✅ DELETE CONFIRMATION
    function confirmDelete(submissionId) {
        if (confirm("⚠️ Are you sure you want to delete this submission? This action cannot be undone!")) {
            window.location.href = "delete_submission.php?id=" + submissionId;
        }
    }

    // ✅ EXPORT CSV FUNCTION
    function exportCSV() {
        window.location.href = "export_csv.php";
    }

    // ✅ EXPORT PDF FUNCTION
    function exportPDF() {
        window.location.href = "export_pdf.php";
    }

    

    // ✅ DEBUG LOGS FOR DEVELOPMENT (Can be removed)
    console.log("📊 Submission Trends Data:", <?= json_encode($submissionTrends) ?>);
    console.log("📊 Status Counts Data:", <?= json_encode($statusCounts) ?>);
    console.log("📊 Top Vehicles Data:", <?= json_encode($topVehicles) ?>);
</script>


<script>
    // ✅ SEARCH FUNCTION
    function searchTable() {
        let input = document.getElementById("search").value.toUpperCase();
        let rows = document.querySelectorAll("#submissionsTable tbody tr");

        rows.forEach(row => {
            let show = Array.from(row.cells).some(cell =>
                cell.textContent.toUpperCase().includes(input)
            );
            row.style.display = show ? "" : "none";
        });
    }

    // ✅ FILTER FUNCTION (JQuery)
    $(document).ready(function () {
        $("#filterStatus").change(function () {
            let selectedStatus = $(this).val().toLowerCase();
            $("#submissionsTable tbody tr").each(function () {
                let rowStatus = $(this).find(".status-dropdown").val().toLowerCase();
                $(this).toggle(selectedStatus === "" || rowStatus === selectedStatus);
            });
        });

        // ✅ STATUS UPDATE AJAX
        $(".status-dropdown").change(function () {
            let submissionId = $(this).data("id");
            let newStatus = $(this).val();
            $.post("update_status.php", { id: submissionId, status: newStatus })
                .done(() => alert("✅ Status updated successfully!"))
                .fail(() => alert("❌ Error updating status!"));
        });
    });

    // ✅ EXPORT FUNCTIONS
    function exportCSV() { window.location.href = "export_csv.php"; }
    function exportPDF() { window.location.href = "export_pdf.php"; }

    // ✅ DELETE CONFIRMATION
    function confirmDelete(submissionId) {
        if (confirm("⚠️ Are you sure you want to delete this submission?")) {
            window.location.href = "delete_submission.php?id=" + submissionId;
        }
    }

    // ✅ NOTIFICATIONS
    document.addEventListener("DOMContentLoaded", function () {
        function loadNotifications() {
            fetch("fetch_notifications.php")
                .then(res => res.json())
                .then(data => {
                    const list = document.getElementById("notification-list");
                    const badge = document.getElementById("notification-count");
                    list.innerHTML = "";
                    data.forEach(n => {
                        let li = document.createElement("li");
                        li.textContent = `${n.message} - ${new Date(n.created_at).toLocaleString()}`;
                        list.appendChild(li);
                    });
                    badge.textContent = data.length;
                    badge.style.display = data.length ? "inline-block" : "none";
                })
                .catch(console.error);
        }
        loadNotifications();
        setInterval(loadNotifications, 30000);
    });

    

    // ✅ AVERAGE SUBMISSIONS
    document.getElementById('avgSubmissions').textContent = Math.round(<?= $totalSubmissions ?> / 30) + " submissions/day";

    // ✅ APPROVAL RATE
    new Chart(document.getElementById('approvalRateChart'), {
        type: 'pie',
        data: {
            labels: ["Approved", "Rejected"],
            datasets: [{
                data: [<?= $approvedSubmissions ?>, <?= $rejectedSubmissions ?>],
                backgroundColor: ["green", "red"]
            }]
        }
    });

  
    // ✅ GLOBAL Chart instances for dynamic updates
const submissionTrendsChart = new Chart(document.getElementById('submissionTrendsChart'), {
    type: 'line',
    data: {
        labels: <?= $submissionTrendsLabels ?>,
        datasets: [{
            label: 'Submissions Per Month',
            data: <?= $submissionTrendsCounts ?>,
            borderColor: 'blue',
            fill: false
        }]
    }
});

const statusPieChart = new Chart(document.getElementById('statusPieChart'), {
    type: 'pie',
    data: {
        labels: <?= $statusLabels ?>,
        datasets: [{
            data: <?= $statusCountsData ?>,
            backgroundColor: ['green', 'orange', 'red']
        }]
    }
});

const topVehiclesChart = new Chart(document.getElementById('topVehiclesChart'), {
    type: 'bar',
    data: {
        labels: <?= $topVehicleLabels ?>,
        datasets: [{
            label: 'Top Vehicles Submitted',
            data: <?= $topVehicleCounts ?>,
            backgroundColor: 'purple'
        }]
    }
});

const equipeSeriesChart = new Chart(document.getElementById('equipeSeriesChart'), {
    type: 'bar',
    data: {
        labels: <?= $equipeSeriesLabels ?>,
        datasets: [{
            label: 'Submissions per Series',
            data: <?= $equipeSeriesCounts ?>,
            backgroundColor: 'goldenrod'
        }]
    },
    options: {
        scales: {
            y: { beginAtZero: true }
        }
    }
});

    
    function applyDateFilter() {
    const start = document.getElementById('startDate').value;
    const end = document.getElementById('endDate').value;

    if (!start || !end) {
        alert("Please select both dates.");
        return;
    }

    fetch("fetch_analytics.php", {
        method: "POST",
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ startDate: start, endDate: end })
    })
    .then(res => res.json())
    .then(data => {
        // ✅ Update each chart's data and re-render
        submissionTrendsChart.data.labels = data.submissionTrends.labels;
        submissionTrendsChart.data.datasets[0].data = data.submissionTrends.counts;
        submissionTrendsChart.update();

        statusPieChart.data.labels = data.statusCounts.labels;
        statusPieChart.data.datasets[0].data = data.statusCounts.counts;
        statusPieChart.update();

        topVehiclesChart.data.labels = data.topVehicles.labels;
        topVehiclesChart.data.datasets[0].data = data.topVehicles.counts;
        topVehiclesChart.update();

        equipeSeriesChart.data.labels = data.equipeSeries.labels;
        equipeSeriesChart.data.datasets[0].data = data.equipeSeries.counts;
        equipeSeriesChart.update();
    });
}

</script>
<script>
function toggleNotifications() {
    const list = document.getElementById("notification-list");
    list.style.display = list.style.display === "block" ? "none" : "block";
}

function loadNotifications() {
    fetch("fetch_notifications.php")
        .then(res => res.json())
        .then(data => {
            const list = document.getElementById("notification-list");
            const count = document.getElementById("notification-count");
            list.innerHTML = "";

            let unread = 0;

            data.forEach(n => {
                const li = document.createElement("li");

                if (n.submission_id) {
                    li.innerHTML = `
                        <a href="view_submission.php?id=${n.submission_id}" style="color:#ccc; text-decoration:none;">
                            ${n.message}
                        </a><br>
                        <small style="color:gray;">${new Date(n.created_at).toLocaleString()}</small>
                    `;
                } else {
                    li.innerHTML = `
                        ${n.message}<br>
                        <small style="color:gray;">${new Date(n.created_at).toLocaleString()}</small>
                    `;
                }

                list.appendChild(li);
                if (!n.is_read) unread++;
            });

            count.textContent = unread;
            count.style.display = unread ? "inline-block" : "none";
        });
}


document.addEventListener("DOMContentLoaded", () => {
    loadNotifications();
    setInterval(loadNotifications, 30000);
});
</script>


</body>
</html>
