<?php
session_start();
if (!isset($_SESSION['admin_logged_in'])) {
    header("Location: admin_login.php");
    exit();
}

require 'db_connect.php';

// Handle filters
$filterStatus = $_GET['filter'] ?? '';
$whereClause = '';
$params = [];

if ($filterStatus === 'unread') {
    $whereClause = 'WHERE is_read = 0';
} elseif ($filterStatus === 'read') {
    $whereClause = 'WHERE is_read = 1';
}

// Pagination setup
$limit = 20;
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$offset = ($page - 1) * $limit;

// Get total count for pagination
$totalStmt = $pdo->query("SELECT COUNT(*) FROM admin_notifications");
$totalNotifications = $totalStmt->fetchColumn();
$totalPages = ceil($totalNotifications / $limit);

// Fetch paginated notifications
$stmt = $pdo->prepare("SELECT * FROM admin_notifications $whereClause ORDER BY created_at DESC LIMIT :limit OFFSET :offset");
if (!empty($whereClause)) {
    foreach ($params as $key => $val) {
        $stmt->bindValue($key, $val);
    }
}
$stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
$stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
$stmt->execute();
$notifications = $stmt->fetchAll(PDO::FETCH_ASSOC);

?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Admin Notifications</title>
    <link rel="stylesheet" href="admin_styles.css">
    <style>
        main {
            width: 80%;
            margin-left: 15%;
            margin-right: 5%;
            padding: 20px;
        }
        .notifications-container {
            width: 100%;
            background: #fff;
            padding: 20px;
            border-radius: 8px;
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
        }
        .notification-item {
            border-bottom: 1px solid #ddd;
            padding: 12px 0;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .notification-item:last-child {
            border-bottom: none;
        }
        .notification-message {
            font-size: 16px;
            flex-grow: 1;
        }
        .notification-time {
            color: #777;
            font-size: 13px;
            margin-right: 10px;
        }
        .notification-actions button {
            margin-left: 5px;
            padding: 4px 8px;
            font-size: 12px;
            cursor: pointer;
        }
        .filters {
            margin-bottom: 15px;
        }
        .pagination {
            margin-top: 20px;
            text-align: center;
        }
        .pagination a {
            margin: 0 5px;
            padding: 6px 10px;
            background: #007bff;
            color: white;
            text-decoration: none;
            border-radius: 4px;
        }
        .pagination a.active {
            background: #0056b3;
        }
    </style>
</head>
<body>
<?php include 'sidebar.php'; ?>

<main>
    <div class="notifications-container">
        <h2>🔔 All Notifications</h2>

        <div class="filters">
            <a href="?filter=all">All</a> |
            <a href="?filter=unread">Unread</a> |
            <a href="?filter=read">Read</a>
        </div>

        <?php if (count($notifications) > 0): ?>
            <?php foreach ($notifications as $note): ?>
                <div class="notification-item">
                    <div class="notification-message">
                        <?php if ($note['submission_id']): ?>
                            <a class="notification-link" href="view_submission.php?id=<?= $note['submission_id'] ?>">
                                <?= htmlspecialchars($note['message']) ?>
                            </a>
                        <?php else: ?>
                            <?= htmlspecialchars($note['message']) ?>
                        <?php endif; ?>
                        <div class="notification-time">
                            <?= date("M d, Y H:i", strtotime($note['created_at'])) ?>
                        </div>
                    </div>
                    <div class="notification-actions">
                        <?php if (!$note['is_read']): ?>
                            <button onclick="markRead(<?= $note['id'] ?>)">Mark as Read</button>
                        <?php endif; ?>
                        <button onclick="deleteNotification(<?= $note['id'] ?>)">Delete</button>
                    </div>
                </div>
            <?php endforeach; ?>
        <?php else: ?>
            <p>No notifications found.</p>
        <?php endif; ?>

        <div class="pagination">
            <?php for ($i = 1; $i <= $totalPages; $i++): ?>
                <a href="?page=<?= $i ?>&filter=<?= $filterStatus ?>" class="<?= $i === $page ? 'active' : '' ?>"><?= $i ?></a>
            <?php endfor; ?>
        </div>
    </div>
</main>

<script>
function markRead(id) {
    fetch("update_status.php", {
        method: "POST",
        headers: { "Content-Type": "application/x-www-form-urlencoded" },
        body: `mark_read_id=${id}`
    }).then(() => location.reload());
}

function deleteNotification(id) {
    if (!confirm("Delete this notification?")) return;
    fetch("update_status.php", {
        method: "POST",
        headers: { "Content-Type": "application/x-www-form-urlencoded" },
        body: `delete_notification_id=${id}`
    }).then(() => location.reload());
}
</script>
</body>
</html>
