document.addEventListener("DOMContentLoaded", function () {
    // Form logic only runs if the car form is present
    const carForm = document.querySelector("#car-form");
    if (carForm) {
        const formSteps = document.querySelectorAll(".form-step");
        const progressBar = document.querySelector(".progress");
        const nextBtns = document.querySelectorAll(".nxt-btn");
        const prevBtns = document.querySelectorAll(".prev-btn");
        let currentStep = 0;

        function updateForm() {
            formSteps.forEach((step, index) => {
                step.classList.toggle("active", index === currentStep);
            });

            if (progressBar) {
                progressBar.style.width = ((currentStep + 1) / formSteps.length) * 100 + "%";
            }
        }

        function validateStep(step) {
            // Only validate visible fields
            const fields = Array.from(step.querySelectorAll("input, select, textarea"))
                .filter(field => field.offsetParent !== null);
            let isValid = true;

            fields.forEach((field) => {
                const isRequired = field.hasAttribute("required") || field.dataset.required === "true";

                // Reset validation message
                field.setCustomValidity("");

                if (isRequired) {
                    const fieldValue = field.value.trim();
                    const isEmpty = !fieldValue || (field.tagName === "SELECT" && fieldValue === "");

                    if (isEmpty) {
                        field.classList.add("invalid");
                        isValid = false;
                    } else {
                        field.classList.remove("invalid");
                    }

                    // Optional: reject non-numeric values in number fields
                    if (field.type === "number" && isNaN(field.value)) {
                        field.classList.add("invalid");
                        field.setCustomValidity("Please enter a valid number.");
                        isValid = false;
                    }
                }
            });

            return isValid;
        }

        nextBtns.forEach((btn) => {
            btn.addEventListener("click", () => {
                const currentFormStep = formSteps[currentStep];
                if (validateStep(currentFormStep)) {
                    currentStep++;
                    updateForm();
                }
            });
        });

        prevBtns.forEach((btn) => {
            btn.addEventListener("click", () => {
                if (currentStep > 0) {
                    currentStep--;
                    updateForm();
                }
            });
        });

        updateForm();
    }

    // Navigation Menu
    const menuToggle = document.getElementById("menu-toggle");
    const mainNav = document.getElementById("main-nav");
    const closeMenu = document.getElementById("close-menu");

    menuToggle?.addEventListener("click", () => {
        mainNav?.classList.toggle("show");
    });

    closeMenu?.addEventListener("click", () => {
        mainNav?.classList.remove("show");
    });

    // Form overlay effects
    const formSection = document.querySelector("#form-section");
    formSection?.addEventListener("click", function () {
        this.classList.add("active");
    });

    document.addEventListener("click", function (event) {
        if (formSection && !formSection.contains(event.target)) {
            formSection.classList.remove("active");
        }
    });

    // Admin panel logic
    const printBtn = document.querySelector(".print-btn");
    if (printBtn) {
        printBtn.addEventListener("click", function () {
            window.print();
        });
    }

    const editBtn = document.querySelector(".edit-btn");
    if (editBtn) {
        const submissionId = editBtn.getAttribute("data-id");
        if (submissionId) {
            window.location.href = `edit_submission.php?id=${submissionId}`;
        }
    }
});
