<?php
session_start();
require 'db_connect.php';

// Check if admin is logged in
if (!isset($_SESSION['admin_logged_in'])) {
    header("Location: admin_login.php");
    exit();
}

// Pagination setup
$limit = 10; // Default submissions per page
$page = isset($_GET['page']) && is_numeric($_GET['page']) ? $_GET['page'] : 1;
$offset = ($page - 1) * $limit;

// Get total number of submissions
$totalQuery = $pdo->query("SELECT COUNT(*) FROM car_submissions");
$totalSubmissions = $totalQuery->fetchColumn();
$totalPages = ceil($totalSubmissions / $limit);

// Fetch submissions for the current page
$query = $pdo->prepare("SELECT * FROM car_submissions ORDER BY submitted_at DESC LIMIT :limit OFFSET :offset");
$query->bindParam(":limit", $limit, PDO::PARAM_INT);
$query->bindParam(":offset", $offset, PDO::PARAM_INT);
$query->execute();
$submissions = $query->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>All Submissions</title>
    <link rel="stylesheet" href="admin_styles.css">
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
</head>
<body>

<?php include 'sidebar.php'; ?> <!-- Sidebar is included here -->

<!-- Main Content -->
<main class="admin-container">
    <h1>All Submissions</h1>

    <!-- Search & Filter -->
    <div class="filter-container">
        <input type="text" id="search" placeholder="Search submissions..." onkeyup="searchTable()">
        
        <label for="filterStatus">Filter by Status:</label>
        <select id="filterStatus" onchange="filterTable()">
            <option value="">All Statuses</option>
            <option value="Pending">Pending</option>
            <option value="Approved">Approved</option>
            <option value="Rejected">Rejected</option>
        </select>

        <button onclick="exportCSV()" style="margin-top:3px">Export CSV</button>
    </div>

    <!-- Submission Table -->
    <table class="admin-table" id="submissionsTable">
        <thead>
            <tr>
                <th onclick="sortTable(0)">ID</th>
                <th onclick="sortTable(1)">Owner Name</th>
                <th onclick="sortTable(2)">Vehicle</th>
                <th onclick="sortTable(3)">Status</th>
                <th onclick="sortTable(4)">Equipe Series</th>
                    <th onclick="sortTable(5)">Equipe Class</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($submissions as $submission): ?>
                <tr>
                    <td><?= htmlspecialchars($submission['id']) ?></td>
                    <td><?= htmlspecialchars($submission['owner_name']) ?></td>
                    <td><?= htmlspecialchars($submission['vehicle_model'])?></td>
                    <td><?= htmlspecialchars($submission['equipe_series']) ?></td>
                    <td><?= htmlspecialchars($submission['equipe_class']) ?></td>
                    
                    <td>
                        <select class="status-dropdown" data-id="<?= $submission['id'] ?>">
                            <option value="Pending" <?= $submission['status'] == 'Pending' ? 'selected' : '' ?>>Pending</option>
                            <option value="Approved" <?= $submission['status'] == 'Approved' ? 'selected' : '' ?>>Approved</option>
                            <option value="Rejected" <?= $submission['status'] == 'Rejected' ? 'selected' : '' ?>>Rejected</option>
                        </select>
                    </td>
                    <td>
                        <a href="view_submission.php?id=<?= $submission['id'] ?>" class="view-btn">View</a>
                        <a href="edit_submission.php?id=<?= $submission['id'] ?>" class="edit-btn">Edit</a>
                        <a href="export_pdf.php?id=<?= $submission['id'] ?>" class="export-btn">Export PDF</a>
                        <a href="javascript:void(0);" class="delete-btn" onclick="confirmDelete(<?= $submission['id'] ?>)">Delete</a>
                    </td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>

    <!-- Pagination -->
    <div class="pagination">
        <?php for ($i = 1; $i <= $totalPages; $i++): ?>
            <a href="?page=<?= $i ?>" class="<?= ($i == $page) ? 'active' : '' ?>"><?= $i ?></a>
        <?php endfor; ?>
    </div>

</main>

<!-- JavaScript -->
<script>
    // 🔍 Search Functionality
    function searchTable() {
        var input = document.getElementById("search").value.toUpperCase();
        var table = document.getElementById("submissionsTable");
        var rows = table.getElementsByTagName("tr");

        for (var i = 1; i < rows.length; i++) {
            var text = rows[i].textContent || rows[i].innerText;
            rows[i].style.display = text.toUpperCase().includes(input) ? "" : "none";
        }
    }

    // 📌 Filter by Status
    function filterTable() {
        var filter = document.getElementById("filterStatus").value.toLowerCase();
        var table = document.getElementById("submissionsTable");
        var rows = table.getElementsByTagName("tr");

        for (var i = 1; i < rows.length; i++) {
            var statusCell = rows[i].getElementsByTagName("td")[3]; 
            if (statusCell) {
                var statusSelect = statusCell.getElementsByTagName("select")[0]; 
                if (statusSelect) {
                    var statusValue = statusSelect.value.toLowerCase(); 
                    rows[i].style.display = (filter === "" || statusValue === filter) ? "" : "none";
                }
            }
        }
    }

    // ✅ Update Status
    $(".status-dropdown").change(function() {
        var submissionId = $(this).data("id");
        var newStatus = $(this).val();

        $.post("update_status.php", { id: submissionId, status: newStatus }, function(response) {
            alert("Status updated!");
        });
    });

    // 🗑️ Delete Submission Confirmation
    function confirmDelete(submissionId) {
        if (confirm("⚠️ Are you sure you want to delete this submission? This action cannot be undone!")) {
            window.location.href = "delete_submission.php?id=" + submissionId;
        }
    }

    // 📤 Export Functions
    function exportCSV() {
        window.location.href = "export_csv.php";
    }
    function exportPDF() {
        window.location.href = "export_pdf.php";
    }
</script>

</body>
</html>
